/**
* \file: e_sdc_ecm_dbus.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: sdc-ecm-engine
*
* \author: Kirill Marinushkin (kmarinushkin@de.adit-jv.com)
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
***********************************************************************/

#include <string.h>
#include <inttypes.h>
#include <openssl/engine.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <agwcl.h>
#include "e_sdc_ecm_log.h"
#include "e_sdc_ecm_err.h"
#include "e_sdc_ecm_dbus.h"

#define DBUS_SESSION_ENV_VAR "DBUS_SESSION_BUS_ADDRESS"

#define DBUS_DEF_BUS_TYPE    "system"
#define DBUS_DEF_BUS_NAME    "com.bosch.AutomotiveProxy"
#define DBUS_DEF_IFACE_NAME  "com.bosch.AutomotiveProxy.CommunicationProtocol"

#define DBUS_METH_CHECK_CERT_NAME    "CheckCert"
#define DBUS_METH_GET_CERT_NAME      "GetDeviceCert"
#define DBUS_METH_SIGN_NAME          "SignWithDeviceKey"

/* DBus setup ctx */
static char *dbus_name_to_path(const char *name, char **path)
{
    size_t i;
    size_t size = strlen(name) + 2;

    *path = realloc(*path, size);
    if (!(*path))
        return NULL;

    snprintf(*path, size, "/%s", name);
    for (i = 0; i < size - 1; i++) {
        if ((*path)[i] == '.')
            (*path)[i] = '/';
    }

    return *path;
}

int sdc_ecm_dbus_set_bus_type(struct sdc_ecm_dbus_ctx *ctx, const char *p)
{
    int rc = 1;

    if (!p) {
        sdc_ecm_err("Invalid DBUs bus type: NULL");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_IFACE_SET_BUS_TYPE,
                SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    free(ctx->bus_type_str);
    ctx->bus_type_str = strdup(p);

    if (!strcmp(p, "system")) {
        ctx->is_bus_type_user = false;
        sdc_ecm_dbg("ecm_bus_type : system");
    } else if (!strcmp(p, "session")) {
        char *dbus_sess_env_var;

        ctx->is_bus_type_user = true;
        sdc_ecm_dbg("ecm_bus_type : session");

        dbus_sess_env_var = getenv(DBUS_SESSION_ENV_VAR);
        if (dbus_sess_env_var == NULL)
            sdc_ecm_inf("%s is unset", DBUS_SESSION_ENV_VAR)
        else
            sdc_ecm_dbg("%s : %s", DBUS_SESSION_ENV_VAR, dbus_sess_env_var);

    } else {
        rc = 0;
        sdc_ecm_err("Invalid DBUs bus type %s, supported: system|session", p);
        SDC_ECMerr(SDC_ECM_F_GATEWAY_IFACE_SET_BUS_TYPE,
                SDC_ECM_R_INVALID_INPUT);
    }

    /* return 1 if succeed or 0 if error */
    return rc;
}

int sdc_ecm_dbus_set_bus_name(struct sdc_ecm_dbus_ctx *ctx, const char *p)
{
    if (!p) {
        sdc_ecm_err("Invalid DBUs bus name: NULL");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_IFACE_SET_BUS_NAME,
                SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    free(ctx->ecm_bus_name);

    ctx->ecm_bus_name = strdup(p);

    /* return 1 if succeed or 0 if error */
    if (ctx->ecm_bus_name) {
        sdc_ecm_dbg("ecm_bus_name : %s", ctx->ecm_bus_name);
        return 1;
    }

    sdc_ecm_dbg("ecm_bus_name unset");
    return 0;
}

int sdc_ecm_dbus_set_iface_name(struct sdc_ecm_dbus_ctx *ctx, const char *p)
{
    if (!p) {
        sdc_ecm_err("Invalid DBUs interface name: NULL");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_IFACE_SET_INTERFACE_NAME,
                SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    free(ctx->ecm_iface_name);

    ctx->ecm_iface_name = strdup(p);

    dbus_name_to_path(ctx->ecm_iface_name, &(ctx->ecm_obj_path));

    /* return 1 if succeed or 0 if error */
    if (ctx->ecm_iface_name) {
        sdc_ecm_dbg("ecm_iface_name : %s", ctx->ecm_iface_name);
        return 1;
    }

    sdc_ecm_dbg("ecm_iface_name unset");
    return 0;
}

/* DBus interaction functions */

/* Debug logging is removed as not supported by sd-bus for DBus clients.
 * For the "name appeared" logging, sd-bus provides a "tracking peers"
 * functionality, which works only for D-Bus services. It is not useful in our
 * use-case, because we act as a DBus client.
 * For other logging, sd-bus does not provide any special functionality.*/

static void dbus_connect(struct sdc_ecm_dbus_ctx *ctx)
{
    int rc;

    if (ctx->bus)
        return;

    rc = (ctx->is_bus_type_user ? sd_bus_open_user(&ctx->bus) :
          sd_bus_open_system(&ctx->bus));
    if (rc < 0) {
        sdc_ecm_err("Failed sd_bus_open_");
        SDC_ECMerr(SDC_ECM_F_DBUS_CONNECT, SDC_ECM_R_DBUS_OPEN_FAILED);
        return;
    }
}

static void dbus_disconnect(struct sdc_ecm_dbus_ctx *ctx)
{
    sd_bus_flush_close_unref(ctx->bus);
    ctx->bus = NULL;
}

static int dbus_call_check_cert(struct sdc_ecm_dbus_ctx *ctx,
        uint8_t *der, int der_len)
{
    int rc = 0;
    int sd_bus_rc;
    sd_bus_error err = SD_BUS_ERROR_NULL;
    sd_bus_message *request = NULL;
    sd_bus_message *reply = NULL;
    uint16_t reply_result;

    /* prepare request */
    sdc_ecm_ext_dbg("DBUS %s request to %s - %s - %s",
                    DBUS_METH_CHECK_CERT_NAME,
                    ctx->ecm_bus_name,
                    ctx->ecm_obj_path,
                    ctx->ecm_iface_name);
    sd_bus_rc = sd_bus_message_new_method_call(ctx->bus, &request,
        ctx->ecm_bus_name, ctx->ecm_obj_path, ctx->ecm_iface_name,
        DBUS_METH_CHECK_CERT_NAME);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_new_method_call");
        SDC_ECMerr(SDC_ECM_F_DBUS_CHECK_CERT, SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sdc_ecm_hexdump_data("der certificate", SDC_ECM_LOGL_VERBOSE_DEBUG,
                         der, der_len);

    sd_bus_rc = sd_bus_message_append_array(request, 'y', der, der_len);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_append_array");
        SDC_ECMerr(SDC_ECM_F_DBUS_CHECK_CERT, SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    /* send */
    sd_bus_rc = sd_bus_call(ctx->bus, request, DBUS_MESSAGE_TIMEOUT_USEC, &err,
                            &reply);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_call_method: %s", err.message);
        SDC_ECMerr(SDC_ECM_F_DBUS_CHECK_CERT, SDC_ECM_R_METHOD_CALL_FAILED);
        goto out;
    }

    /* parse reply values */
    sd_bus_rc = sd_bus_message_read(reply, "q", &reply_result);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_read");
        SDC_ECMerr(SDC_ECM_F_DBUS_CHECK_CERT, SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sdc_ecm_ext_dbg("DBUS reply_result %u", reply_result);

    if (reply_result != 0) {
        sdc_ecm_err("Gateway returned error code: %d", reply_result);
        SDC_ECMerr(SDC_ECM_F_DBUS_CHECK_CERT,
                SDC_ECM_R_GATEWAY_CERT_CHECK_FAILED);
        goto out;
    }

    rc = 1;

out:
    sd_bus_error_free(&err);
    sd_bus_message_unref(request);
    sd_bus_message_unref(reply);

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int dbus_call_get_cert(struct sdc_ecm_dbus_ctx *ctx, uint64_t flags,
        uint8_t **out_data, int *out_len)
{
    int rc = 0;
    int sd_bus_rc;
    sd_bus_error err = SD_BUS_ERROR_NULL;
    sd_bus_message *request = NULL;
    sd_bus_message *reply = NULL;
    uint16_t reply_result;
    const void *reply_data = NULL;
    size_t reply_len;

    *out_len = 0;
    *out_data = NULL;

    /* prepare request */
    sdc_ecm_ext_dbg("DBUS %s request to %s - %s - %s - flags 0x%"PRIx64,
                    DBUS_METH_GET_CERT_NAME,
                    ctx->ecm_bus_name,
                    ctx->ecm_obj_path,
                    ctx->ecm_iface_name,
                    flags);
    sd_bus_rc = sd_bus_message_new_method_call(ctx->bus, &request,
        ctx->ecm_bus_name, ctx->ecm_obj_path, ctx->ecm_iface_name,
        DBUS_METH_GET_CERT_NAME);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_new_method_call");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sd_bus_rc = sd_bus_message_append(request, "t", flags);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_append");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    /* send */
    sd_bus_rc = sd_bus_call(ctx->bus, request, DBUS_MESSAGE_TIMEOUT_USEC, &err,
                            &reply);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_call_method: %s", err.message);
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY, SDC_ECM_R_METHOD_CALL_FAILED);
        goto out;
    }

    /* parse reply values */
    sd_bus_rc = sd_bus_message_read_array(reply, 'y', &reply_data, &reply_len);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_read_array");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sd_bus_rc = sd_bus_message_read(reply, "q", &reply_result);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_read");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sdc_ecm_ext_dbg("DBUS reply_result %u", reply_result);

    if (reply_result != 0) {
        sdc_ecm_err("Gateway returned error code: %d", reply_result);
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                   SDC_ECM_R_GATEWAY_CERT_CHECK_FAILED);
        goto out;
    }

    sdc_ecm_hexdump_data("certificate", SDC_ECM_LOGL_VERBOSE_DEBUG,
                         reply_data, reply_len);

    /* copy reply to output arguments */
    if (reply_len) {
        *out_len = (int)reply_len;

        *out_data = malloc(reply_len);
        if (!*out_data) {
            SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY, SDC_ECM_R_NO_MEM);
            goto out;
        }

        memcpy(*out_data, reply_data, reply_len);
    }

    rc = 1;

out:
    sd_bus_error_free(&err);
    sd_bus_message_unref(request);
    sd_bus_message_unref(reply);

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int dbus_call_sign(struct sdc_ecm_dbus_ctx *ctx, uint64_t flags,
        const uint8_t *in_data, size_t in_len,
        uint8_t *out_data, size_t *out_len)
{
    int rc = 0;
    int sd_bus_rc;
    sd_bus_error err = SD_BUS_ERROR_NULL;
    sd_bus_message *request = NULL;
    sd_bus_message *reply = NULL;
    uint16_t reply_result;
    const void *reply_data = NULL;
    size_t reply_len;

    *out_len = 0;

    /* prepare request */
    sdc_ecm_ext_dbg("DBUS %s request to %s - %s - %s - flags 0x%"PRIx64,
                    DBUS_METH_SIGN_NAME,
                    ctx->ecm_bus_name,
                    ctx->ecm_obj_path,
                    ctx->ecm_iface_name,
                    flags);
    sd_bus_rc = sd_bus_message_new_method_call(ctx->bus, &request,
        ctx->ecm_bus_name, ctx->ecm_obj_path, ctx->ecm_iface_name,
        DBUS_METH_SIGN_NAME);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_new_method_call");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sdc_ecm_hexdump_data("data to sign", SDC_ECM_LOGL_VERBOSE_DEBUG,
                         in_data, in_len);

    sd_bus_rc = sd_bus_message_append_array(request, 'y', in_data, in_len);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_append_array");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sd_bus_rc = sd_bus_message_append(request, "t", flags);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_append_array");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN,
                   SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    /* send */
    sd_bus_rc = sd_bus_call(ctx->bus, request, DBUS_MESSAGE_TIMEOUT_USEC, &err,
                            &reply);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_call_method: %s", err.message);
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN, SDC_ECM_R_METHOD_CALL_FAILED);
        goto out;
    }
    /* parse reply values */
    sd_bus_rc = sd_bus_message_read_array(reply, 'y', &reply_data, &reply_len);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_read_array");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN, SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sd_bus_rc = sd_bus_message_read(reply, "q", &reply_result);
    if (sd_bus_rc < 0) {
        sdc_ecm_err("Failed sd_bus_message_read");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN, SDC_ECM_R_PARAMETER_CAST_FAILED);
        goto out;
    }

    sdc_ecm_ext_dbg("DBUS reply_result %u", reply_result);

    if (reply_result != 0) {
        sdc_ecm_err("Gateway returned error code: %d", reply_result);
        SDC_ECMerr(SDC_ECM_F_GATEWAY_SIGN,
                SDC_ECM_R_GATEWAY_CERT_CHECK_FAILED);
        goto out;
    }

    sdc_ecm_hexdump_data("signature", SDC_ECM_LOGL_VERBOSE_DEBUG,
                         reply_data, reply_len);

    /* copy reply to output arguments */
    if (reply_len) {
        *out_len = (int)reply_len;

        /* the check of the output buffer size is not supported by openssl
         * engine API */
        memcpy(out_data, reply_data, reply_len);
    }

    rc = 1;

out:
    sd_bus_error_free(&err);
    sd_bus_message_unref(request);
    sd_bus_message_unref(reply);

    /* return 1 if succeed or 0 if error */
    return rc;
}

static EVP_PKEY *gen_pkey(ENGINE *e)
{
    EVP_PKEY *pkey = NULL;
    RSA *rsa = RSA_new_method(e);

    if (!rsa)
         return NULL;

    /* RSA components should be non-NULL */
    if ((rsa->n = BN_new()) == NULL)
        return NULL;
    if ((rsa->d = BN_new()) == NULL)
        return NULL;
    if ((rsa->e = BN_new()) == NULL)
        return NULL;
    if ((rsa->p = BN_new()) == NULL)
        return NULL;
    if ((rsa->q = BN_new()) == NULL)
        return NULL;
    if ((rsa->dmp1 = BN_new()) == NULL)
        return NULL;
    if ((rsa->dmq1 = BN_new()) == NULL)
        return NULL;
    if ((rsa->iqmp = BN_new()) == NULL)
        return NULL;

    pkey = EVP_PKEY_new();
    if (!pkey)
        return NULL;

    if (!EVP_PKEY_set1_RSA(pkey, rsa)) {
        EVP_PKEY_free(pkey);
        pkey = NULL;
    }

    return pkey;
}

static int gen_pubkey(X509 *cert)
{
    int rc = 0;
    EVP_PKEY *tmpkey = NULL;
    X509_REQ *req = NULL;

    cert->cert_info = X509_CINF_new();

    if (!cert->cert_info) {
        sdc_ecm_err("gen_pubkey failed X509_CINF_new");
        return 0;
    }

    req = X509_REQ_new();
    if (!req) {
        sdc_ecm_err("gen_pubkey failed X509_REQ_new");
        return 0;
    }

    tmpkey = X509_REQ_get_pubkey(req);
    if (!tmpkey) {
        sdc_ecm_err("gen_pubkey failed X509_REQ_get_pubkey");
        goto out;
    }

    if (!X509_set_pubkey(cert, tmpkey)) {
        sdc_ecm_err("gen_pubkey failed X509_set_pubkey");
        goto out;
    }

    rc = 1;

out:
    X509_REQ_free(req);
    EVP_PKEY_free(tmpkey);

    /* return 1 if succeed or 0 if error */
    return rc;
}

static int set_meths(ENGINE *e, X509 *cert, EVP_PKEY *key)
{
    EVP_PKEY *pubkey_p = X509_get_pubkey(cert);

    if (!pubkey_p) {
        if (!gen_pubkey(cert)) {
            sdc_ecm_err("set_meths failed to generate pubkey_p");
            return 0;
        }

        pubkey_p = X509_get_pubkey(cert);
        if (!pubkey_p) {
            sdc_ecm_err("set_meths invalid pubkey_p");
            return 0;
        }
    }

    /* associate private and public keys */
    pubkey_p->type = key->type;
    pubkey_p->pkey = key->pkey;

    /* configure private key */
    key->engine = e;

    /* configure public key */
    pubkey_p->engine = e;
    pubkey_p->ameth =
            ENGINE_get_pkey_asn1_meth_str(e, "sdc-ecm-asn1-pub", -1);

    /* return 1 if succeed or 0 if error */
    return 1;
}

/* Functions called by engine */
int sdc_ecm_dbus_init(struct sdc_ecm_dbus_ctx *ctx,
        struct sdc_ecm_key_ctx *key_ctx)
{
    /* init dbus ctx */
    ctx->bus = NULL;
    ctx->is_bus_type_user = false;
    ctx->is_server_cert_checked = false;
    ctx->is_server_cert_check_skippable = false;
    pthread_mutex_init(&ctx->mut_dbus, NULL);

    if (!sdc_ecm_dbus_set_bus_type(ctx, DBUS_DEF_BUS_TYPE))
        return 0;
    if (!sdc_ecm_dbus_set_bus_name(ctx, DBUS_DEF_BUS_NAME))
        return 0;
    if (!sdc_ecm_dbus_set_iface_name(ctx, DBUS_DEF_IFACE_NAME))
        return 0;

    /* init key ctx */
    key_ctx->ready = 0;
    key_ctx->cert = NULL;
    key_ctx->pkey = NULL;

    /* init tohash list head */
    ctx->tohash_head = NULL;
    pthread_mutex_init(&ctx->mut_hash, NULL);

    /* return 1 if succeed or 0 if error */
    return 1;
}

void sdc_ecm_dbus_finish(struct sdc_ecm_dbus_ctx *ctx,
        struct sdc_ecm_key_ctx *key_ctx)
{
    ctx->is_server_cert_checked = false;

    sdc_ecm_dbus_cleanup_key_pair(key_ctx);
}

#ifdef NO_SDC_PREHASHED_SIGN
/* work with tohash_list */
static void tohash_list_cleanup_timedout(struct sdc_ecm_dbus_ctx *ctx)
{
    struct sdc_ecm_hash_ctx *parent = NULL;
    struct sdc_ecm_hash_ctx *node;
    time_t cur_time = time(NULL);

    sdc_ecm_ext_dbg("enter %s", __FUNCTION__);

    node = ctx->tohash_head;

    while (node) {
        if (node->timestamp < cur_time) {
            struct sdc_ecm_hash_ctx *tofree = node;

            if (parent) {
                parent->next = node->next;
            } else {
                ctx->tohash_head = node->next;
            }

            node = node->next;

            sdc_ecm_ext_dbg("%s, %p", __FUNCTION__, tofree);

            free(tofree->data);
            free(tofree);

            continue;
        }

        parent = node;
        node = node->next;
    }
}

static struct sdc_ecm_hash_ctx *tohash_list_add(struct sdc_ecm_dbus_ctx *ctx)
{
    struct sdc_ecm_hash_ctx *node;

    sdc_ecm_ext_dbg("enter %s", __FUNCTION__);

    tohash_list_cleanup_timedout(ctx);

    node = ctx->tohash_head;
    if (!node) {
        ctx->tohash_head = calloc(1, sizeof(struct sdc_ecm_hash_ctx));
        if (ctx->tohash_head)
            ctx->tohash_head->timestamp = time(NULL) + HASH_TIMEOUT;

        sdc_ecm_ext_dbg("%s, new head %p", __FUNCTION__, ctx->tohash_head);

        return ctx->tohash_head;
    }

    while (node->next)
        node = node->next;

    node->next = calloc(1, sizeof(struct sdc_ecm_hash_ctx));
    if (node->next)
        node->next->timestamp = time(NULL) + HASH_TIMEOUT;

    sdc_ecm_ext_dbg("%s, %p", __FUNCTION__, node->next);

    return node->next;
}

static struct sdc_ecm_hash_ctx *tohash_list_get(struct sdc_ecm_dbus_ctx *ctx,
    EVP_MD_CTX *md_ctx)
{
    struct sdc_ecm_hash_ctx *node = ctx->tohash_head;

    sdc_ecm_ext_dbg("enter %s", __FUNCTION__);

    while (node) {
        if (node->md_ctx == md_ctx)
            break;

        node = node->next;
    }

    sdc_ecm_ext_dbg("%s, %p", __FUNCTION__, node);

    return node;
}

static struct sdc_ecm_hash_ctx *tohash_list_get_ready(
    struct sdc_ecm_dbus_ctx *ctx, const uint8_t *in_data, size_t in_len)
{
    struct sdc_ecm_hash_ctx *node = ctx->tohash_head;

    sdc_ecm_ext_dbg("enter %s", __FUNCTION__);

    while (node) {
        if ((node->ready) &&
            (node->hash_len == in_len) &&
            !memcmp(node->hash, in_data, in_len)) {
            /* node found */
            break;
        }

        node = node->next;
    }

    sdc_ecm_ext_dbg("%s, %p", __FUNCTION__, node);

    return node;
}

static void tohash_list_remove_ready(struct sdc_ecm_dbus_ctx *ctx,
    const uint8_t *in_data, size_t in_len)
{
    struct sdc_ecm_hash_ctx *parent = NULL;
    struct sdc_ecm_hash_ctx *node = ctx->tohash_head;

    sdc_ecm_ext_dbg("enter %s", __FUNCTION__);

    while (node) {
        if ((node->ready) &&
            (node->hash_len == in_len) &&
            ((node->hash == in_data) ||
                !memcmp(node->hash, in_data, in_len))) {
            struct sdc_ecm_hash_ctx *tofree = node;

            if (parent) {
                parent->next = node->next;
            } else {
                ctx->tohash_head = NULL;
            }

            sdc_ecm_ext_dbg("%s, %p", __FUNCTION__, tofree);

            free(tofree->data);
            free(tofree);

            break;
        }

        parent = node;
        node = node->next;
    }
}

int sdc_ecm_dbus_hash_init(EVP_MD_CTX *md_ctx, struct sdc_ecm_dbus_ctx *ctx,
        size_t digest_len)
{
    int rc = 0;
    struct sdc_ecm_hash_ctx *node;

    pthread_mutex_lock(&ctx->mut_hash);

    node = tohash_list_get(ctx, md_ctx);
    if (node) {
        /* already existing - remove it */
        node->ready = 1;
        tohash_list_remove_ready(ctx, node->hash, node->hash_len);
    }

    /* create new */
    node = tohash_list_add(ctx);
    if (!node)
        goto out;

    node->md_ctx = md_ctx;
    node->hash_len = digest_len;

    rc = 1;

out:
    pthread_mutex_unlock(&ctx->mut_hash);

    /* return 1 if succeed or 0 if error */
    return rc;
}

int sdc_ecm_dbus_hash_update(EVP_MD_CTX *md_ctx, struct sdc_ecm_dbus_ctx *ctx,
        const void *data, size_t count)
{
    int rc = 0;
    struct sdc_ecm_hash_ctx *node;

    pthread_mutex_lock(&ctx->mut_hash);

    node = tohash_list_get(ctx, md_ctx);
    if (!node)
        goto out;

    node->data = realloc(node->data, node->len + count);
    if (!node->data) {
        sdc_ecm_err("No memory to reallocate data-to-hash");
        SDC_ECMerr(SDC_ECM_F_DIGEST, SDC_ECM_R_NO_MEM);
        goto out;
    }

    memcpy(&node->data[node->len], data, count);
    node->len += count;

    rc = 1;

out:
    pthread_mutex_unlock(&ctx->mut_hash);

    /* return 1 if succeed or 0 if error */
    return rc;
}

int sdc_ecm_dbus_hash_final(EVP_MD_CTX *md_ctx, struct sdc_ecm_dbus_ctx *ctx,
        uint8_t *md)
{
    int rc = 0;
    struct sdc_ecm_hash_ctx *node;

    pthread_mutex_lock(&ctx->mut_hash);

    node = tohash_list_get(ctx, md_ctx);
    if (!node)
        goto out;

    memcpy(node->hash, md, node->hash_len);
    node->ready = 1;

    rc = 1;

out:
    pthread_mutex_unlock(&ctx->mut_hash);

    /* return 1 if succeed or 0 if error */
    return rc;
}

int sdc_ecm_dbus_hash_copy(EVP_MD_CTX *to, EVP_MD_CTX *from,
        struct sdc_ecm_dbus_ctx *ctx)
{
    int rc = 0;
    struct sdc_ecm_hash_ctx *node_from;
    struct sdc_ecm_hash_ctx *node_to;

    pthread_mutex_lock(&ctx->mut_hash);

    node_from = tohash_list_get(ctx, from);
    if (!node_from)
        goto out;

    /* create new */
    node_to = tohash_list_add(ctx);
    if (!node_to)
        goto out;

    node_to->md_ctx = to;
    node_to->ready = node_from->ready;
    node_to->len = node_from->len;

    node_to->data = malloc(node_to->len);
    if  (!node_to->data)
        goto out;

    memcpy(node_to->data, node_from->data, node_to->len);

    node_to->hash_len = node_from->hash_len;
    memcpy(node_to->hash, node_from->hash, node_to->hash_len);

    rc = 1;

out:
    pthread_mutex_unlock(&ctx->mut_hash);

    /* return 1 if succeed or 0 if error */
    return rc;
}
#endif /* NO_SDC_PREHASHED_SIGN */

int sdc_ecm_dbus_check_server_cert(struct sdc_ecm_dbus_ctx *ctx, X509 *cert)
{
    int rc = 0;
    uint8_t *der = NULL;
    int der_len = i2d_X509(cert, &der);

    if (!der || der_len <= 0) {
        sdc_ecm_err("Failed to get DER from X509 certificate");
        SDC_ECMerr(SDC_ECM_F_DBUS_CHECK_CERT, SDC_ECM_R_INVALID_INPUT);
        return 0;
    }

    pthread_mutex_lock(&ctx->mut_dbus);

    ctx->is_server_cert_checked = false;

    dbus_connect(ctx);

    if (ctx->bus) {
        rc = dbus_call_check_cert(ctx, der, der_len);

        dbus_disconnect(ctx);
    }

    if (rc)
        ctx->is_server_cert_checked = true;

    pthread_mutex_unlock(&ctx->mut_dbus);

    /* return 1 if succeed or 0 if error */
    return rc;
}

int sdc_ecm_dbus_sign(struct sdc_ecm_dbus_ctx *ctx, int md_type,
        const uint8_t *in_data, size_t in_len,
        uint8_t *out_data, size_t *out_len) {
    uint64_t flags = AGWCL_F_SIG_RSA | AGWCL_F_SIG_PKCS1 | AGWCL_F_SIG_DER;
    int rc = 0;
    const uint8_t *tosign_data = in_data;
    size_t tosign_len = in_len;
#ifdef NO_SDC_PREHASHED_SIGN
    struct sdc_ecm_hash_ctx *node =
        tohash_list_get_ready(ctx, in_data, in_len);

    /* check data to be signed */
    if (node) {
        tosign_data = node->data;
        tosign_len = node->len;
    } else {
        tosign_data = in_data;
        tosign_len = in_len;
    }
#else
    flags |= AGWCL_F_SIG_PRECOMPUTED_HASH;
#endif /* NO_SDC_PREHASHED_SIGN */

    pthread_mutex_lock(&ctx->mut_dbus);

    /* prepare flags */
    switch(md_type) {
    case NID_md5:
        flags |= AGWCL_F_SIG_MD5;
        break;
    case NID_sha1:
        flags |= AGWCL_F_SIG_SHA1;
        break;
    case NID_sha224:
        flags |= AGWCL_F_SIG_SHA224;
        break;
    case NID_sha256:
        flags |= AGWCL_F_SIG_SHA256;
        break;
    case NID_sha384:
        flags |= AGWCL_F_SIG_SHA384;
        break;
    case NID_sha512:
        flags |= AGWCL_F_SIG_SHA512;
        break;
    }

    sdc_ecm_dbg("flags=0x%lx", (unsigned long)flags);

    dbus_connect(ctx);

    if (ctx->bus) {
        rc = dbus_call_sign(ctx, flags, tosign_data, tosign_len,
                out_data, out_len);

        dbus_disconnect(ctx);
    }

    pthread_mutex_unlock(&ctx->mut_dbus);

#ifdef NO_SDC_PREHASHED_SIGN
    tohash_list_remove_ready(ctx, in_data, in_len);
#endif /* NO_SDC_PREHASHED_SIGN */

    /* return 1 if succeed or 0 if error */
    return rc;
}

int sdc_ecm_dbus_load_key_pair(ENGINE *e, struct sdc_ecm_dbus_ctx *ctx,
        struct sdc_ecm_key_ctx *key_ctx)
{
    int rc = 0;
    uint8_t *pem = NULL;
    int pem_len = 0;
    uint64_t flags = AGWCL_F_CERT_PEM;
    BIO *bp = NULL;

    if (!ctx || !e || !key_ctx) {
        sdc_ecm_err("Invalid input: sdc_ecm_dbus_load_key_pair");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                SDC_ECM_R_INVALID_INPUT);

        return 0;
    }

    sdc_ecm_ext_dbg(__FUNCTION__);

    pthread_mutex_lock(&ctx->mut_dbus);

    sdc_ecm_dbus_cleanup_key_pair(key_ctx);

    dbus_connect(ctx);

    if (ctx->bus) {
        rc = dbus_call_get_cert(ctx, flags, &pem, &pem_len);

        dbus_disconnect(ctx);
    }

    if (rc != 0) { /* dbus call successful */
        rc = 0; /* reset error code */

        bp = BIO_new_mem_buf(pem, pem_len);
        if (!bp) {
            sdc_ecm_err("Failed to get BIO mem buffer");
            SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                       SDC_ECM_R_NO_MEM);
            goto out;
        }

        key_ctx->cert = PEM_read_bio_X509(bp, NULL, NULL, NULL);

        BIO_free(bp);
    }

    if (!key_ctx->cert) {
        sdc_ecm_err("Failed to get X509 certificate");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                SDC_ECM_R_GATEWAY_GET_CERTIFICATE_FAILED);
        goto out;
    }

    key_ctx->pkey = gen_pkey(e);
    if (!key_ctx->pkey) {
        sdc_ecm_err("Failed to get private key handler");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                SDC_ECM_R_FAILED_TO_GET_KEY_HANDLER);
        goto out;
    }

    rc = set_meths(e, key_ctx->cert, key_ctx->pkey);
    if (!rc) {
        sdc_ecm_err("Failed to set meths");
        SDC_ECMerr(SDC_ECM_F_GATEWAY_LOAD_KEY,
                SDC_ECM_R_FAILED_TO_SET_KEY_METHODS);
        goto out;
    }

    key_ctx->ready = 1;

out:
    pthread_mutex_unlock(&ctx->mut_dbus);

    /* return 1 if succeed or 0 if error */
    return rc;
}

void sdc_ecm_dbus_cleanup_key_pair(struct sdc_ecm_key_ctx *key_ctx)
{
    if (!key_ctx)
        return;

    sdc_ecm_ext_dbg(__FUNCTION__);

    key_ctx->ready = 0;

    X509_free(key_ctx->cert);
    key_ctx->cert = NULL;

    EVP_PKEY_free(key_ctx->pkey);
    key_ctx->pkey = NULL;
}
